package com.atlassian.plugin.webresource;

import com.atlassian.fugue.Option;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.cache.filecache.FileCache;
import com.atlassian.plugin.cache.filecache.impl.FileCacheImpl;
import com.atlassian.plugin.cache.filecache.impl.PassThroughFileCache;
import com.atlassian.plugin.elements.ResourceDescriptor;
import com.atlassian.plugin.event.PluginEventListener;
import com.atlassian.plugin.event.PluginEventManager;
import com.atlassian.plugin.event.events.PluginDisabledEvent;
import com.atlassian.plugin.event.events.PluginEnabledEvent;
import com.atlassian.plugin.event.events.PluginModuleDisabledEvent;
import com.atlassian.plugin.event.events.PluginModuleEnabledEvent;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.servlet.ServletContextFactory;
import com.atlassian.plugin.util.PluginUtils;
import com.atlassian.plugin.webresource.cache.FileCacheKey;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.data.DefaultPluginDataResource;
import com.atlassian.plugin.webresource.transformer.DefaultStaticTransformers;
import com.atlassian.plugin.webresource.transformer.DefaultStaticTransformersSupplier;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.plugin.webresource.url.DefaultUrlBuilderMap;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.atlassian.webresource.api.data.PluginDataResource;
import com.atlassian.webresource.api.data.WebResourceDataProvider;
import com.google.common.base.Function;
import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicLong;

import static com.atlassian.plugin.webresource.AbstractBatchResourceBuilder.skipBatch;

/**
 * Default implementation of {@link PluginResourceLocator}.
 *
 * @since 2.2
 */
public class PluginResourceLocatorImpl implements PluginResourceLocator
{
    private static final Logger log = LoggerFactory.getLogger(PluginResourceLocatorImpl.class);

    final private PluginAccessor pluginAccessor;
    final private TransformerCache transformerCache;
    final private StaticTransformers staticTransformers;
    final private WebResourceUrlProvider webResourceUrlProvider;
    final private ResourceBatchingConfiguration batchingConfiguration;
    final private List<DownloadableResourceBuilder> builders;
    private final FileCache<FileCacheKey> fileCache;

    /**
     * This field is static to allow multiple instances of {@link FileCacheImpl} in the system to use the same tmp dir,
     * without overwriting each other's files.
     */
    private static final AtomicLong FILENAME_COUNTER = new AtomicLong(0);

    static final String RESOURCE_SOURCE_PARAM = "source";
    static final String RESOURCE_BATCH_PARAM = "batch";
    /** there can easily be 50-80 in a "normal" app, make it 20x bigger */
    static final int DEFAULT_CACHE_SIZE = 1000;

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
                new DefaultResourceBatchingConfiguration(), pluginEventManager);
    }

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceBatchingConfiguration batchingConfiguration, final PluginEventManager pluginEventManager)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
                new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration),
                batchingConfiguration, pluginEventManager,
                new DefaultStaticTransformers(new DefaultStaticTransformersSupplier(webResourceIntegration, webResourceUrlProvider)));
    }

    public PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceBatchingConfiguration batchingConfiguration, final PluginEventManager pluginEventManager,
            final StaticTransformers staticTransformers)
    {
        this(webResourceIntegration, servletContextFactory, webResourceUrlProvider,
                new DefaultResourceDependencyResolver(webResourceIntegration, batchingConfiguration),
                batchingConfiguration, pluginEventManager, staticTransformers);
    }

    private PluginResourceLocatorImpl(final WebResourceIntegration webResourceIntegration,
            final ServletContextFactory servletContextFactory, final WebResourceUrlProvider webResourceUrlProvider,
            final ResourceDependencyResolver dependencyResolver, final ResourceBatchingConfiguration batchingConfiguration,
            final PluginEventManager pluginEventManager, final StaticTransformers staticTransformers)
    {
        this.pluginAccessor = webResourceIntegration.getPluginAccessor();
        this.staticTransformers = staticTransformers;
        this.webResourceUrlProvider = webResourceUrlProvider;
        this.batchingConfiguration = batchingConfiguration;

        FileCache<FileCacheKey> cache = makePhysicalCache(webResourceIntegration);
        if (cache == null) {
            cache = PassThroughFileCache.instance();
        }

        this.fileCache = cache;
        this.transformerCache = new TransformerCache(pluginEventManager, pluginAccessor);

        final SingleDownloadableResourceBuilder singlePluginBuilder =
                new SingleDownloadableResourceBuilder(pluginAccessor, transformerCache, staticTransformers, servletContextFactory);

        builders = Collections.unmodifiableList(Arrays.asList(
                new SuperBatchDownloadableResourceBuilder(dependencyResolver, pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                new SuperBatchSubResourceBuilder(dependencyResolver, singlePluginBuilder),
                new ContextBatchDownloadableResourceBuilder(dependencyResolver, pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                new ContextBatchSubResourceBuilder(dependencyResolver, singlePluginBuilder),
                new SingleBatchDownloadableResourceBuilder(pluginAccessor, webResourceUrlProvider, singlePluginBuilder, cache, batchingConfiguration),
                singlePluginBuilder));

        // TODO find better way to register, move it outside of the constructor.
        pluginEventManager.register(this);
    }

    private static FileCache<FileCacheKey> makePhysicalCache(WebResourceIntegration webResourceIntegration) {

        boolean cacheDisabled = Boolean.getBoolean(PluginUtils.WEBRESOURCE_DISABLE_FILE_CACHE) || Boolean.getBoolean(PluginUtils.ATLASSIAN_DEV_MODE);
        if (cacheDisabled) {
            return null;
        }


        FileCache<FileCacheKey> cache = null;
        try {
            int cachesize = Integer.getInteger(PluginUtils.WEBRESOURCE_FILE_CACHE_SIZE, DEFAULT_CACHE_SIZE);
            final File tmpDir = webResourceIntegration.getTemporaryDirectory();
            if (tmpDir != null) {
                cache = new FileCacheImpl<FileCacheKey>(tmpDir, cachesize, FILENAME_COUNTER);
            }
        } catch (IOException e) {
            log.error("Could not create file cache object, will startup with filecaching disabled, please investigate the cause and correct it.", e);
        }
        return cache;
    }

    @PluginEventListener
    public void onPluginDisabled(final PluginDisabledEvent event)
    {
        fileCache.clear();
    }

    @PluginEventListener
    public void onPluginEnabled(final PluginEnabledEvent event)
    {
        fileCache.clear();
    }

    @PluginEventListener
    public void onPluginModuleEnabled(final PluginModuleEnabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
            fileCache.clear();
    }

    @PluginEventListener
    public void onPluginModuleDisabled(final PluginModuleDisabledEvent event)
    {
        if (event.getModule() instanceof WebResourceModuleDescriptor)
            fileCache.clear();
    }

    public boolean matches(final String url)
    {
        for (final DownloadableResourceBuilder builder : builders)
        {
            if (builder.matches(url))
            {
                return true;
            }
        }

        return false;
    }

    public DownloadableResource getDownloadableResource(final String url, final Map<String, String> queryParams)
    {
        try
        {
            for (final DownloadableResourceBuilder builder : builders)
            {
                if (builder.matches(url))
                {
                    return builder.parse(url, queryParams);
                }
            }
        }
        catch (final UrlParseException e)
        {
            log.error("Unable to parse URL: " + url, e);
        }
        // TODO: It would be better to use Exceptions rather than returning
        // nulls to indicate an error.
        return null;
    }

    @Override
    public Iterable<PluginDataResource> getPluginDataResources(final String moduleCompleteKey)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        return getPluginDataResources(option.get());
    }

    @Override
    public Iterable<PluginDataResource> getPluginDataResources(final WebResourceModuleDescriptor moduleDescriptor)
    {
        return Iterables.transform(moduleDescriptor.getDataProviders().entrySet(), new Function<Map.Entry<String, WebResourceDataProvider>, PluginDataResource>()
        {
            @Override
            public PluginDataResource apply(Map.Entry<String, WebResourceDataProvider> input)
            {
                return new DefaultPluginDataResource(moduleDescriptor.getCompleteKey() + "." + input.getKey(), input.getValue().get());
            }
        });
    }

    @Override
    public List<PluginResource> getPluginResources(final String moduleCompleteKey, ConditionState conditionsRun)
    {
        Option<WebResourceModuleDescriptor> option = getDescriptor(moduleCompleteKey);
        if (option.isEmpty())
        {
            return Collections.emptyList();
        }

        WebResourceModuleDescriptor wrmd = option.get();
        return getPluginResources(wrmd, conditionsRun);
    }

    @Override
    public List<PluginResource> getPluginResources(WebResourceModuleDescriptor wrmd,
                                                   ConditionState conditionsRun)
    {
        final boolean singleMode = !batchingConfiguration.isPluginWebResourceBatchingEnabled();
        final Set<PluginResource> resources = new LinkedHashSet<PluginResource>();

        DefaultUrlBuilderMap urlBuilderMap = new DefaultUrlBuilderMap(wrmd, transformerCache, staticTransformers);

        for (final ResourceDescriptor resourceDescriptor : wrmd.getResourceDescriptors())
        {
            String sourceType = ResourceUtils.getType(resourceDescriptor.getLocation());
            String destinationType = ResourceUtils.getType(resourceDescriptor.getName());
            UrlParameters urlParams = urlBuilderMap.getOrCreateForType(sourceType, conditionsRun);

            if (singleMode || skipBatch(resourceDescriptor))
            {
                final boolean cache = !"false".equalsIgnoreCase(resourceDescriptor.getParameter("cache"));
                resources.add(new SinglePluginResource(resourceDescriptor.getName(), wrmd.getCompleteKey(),
                    cache, urlParams, resourceDescriptor.getParameters()));
            }
            else
            {
                final BatchPluginResource batchResource = createBatchResource(wrmd.getCompleteKey(), resourceDescriptor,
                        destinationType, urlParams,
                        new BatchedWebResourceDescriptor(wrmd.getPlugin().getPluginInformation().getVersion(), wrmd.getCompleteKey()));
                resources.add(batchResource);
            }
        }
        return ImmutableList.copyOf(resources);
    }

    @Override
    public Option<WebResourceModuleDescriptor> getDescriptor(String moduleCompleteKey)
    {
        final ModuleDescriptor<?> moduleDescriptor = pluginAccessor.getEnabledPluginModule(moduleCompleteKey);
        if ((moduleDescriptor == null) || !(moduleDescriptor instanceof WebResourceModuleDescriptor))
        {
            log.error("Error loading resource \"{}\". Resource is not a Web Resource Module", moduleCompleteKey);
            return Option.none();
        }
        return Option.some((WebResourceModuleDescriptor) moduleDescriptor);
    }

    // package protected so we can test it
    String[] splitLastPathPart(final String resourcePath)
    {
        int indexOfSlash = resourcePath.lastIndexOf('/');
        if (resourcePath.endsWith("/")) // skip over the trailing slash
        {
            indexOfSlash = resourcePath.lastIndexOf('/', indexOfSlash - 1);
        }

        if (indexOfSlash < 0)
        {
            return null;
        }

        return new String[] {resourcePath.substring(0, indexOfSlash + 1), resourcePath.substring(indexOfSlash + 1)};
    }

    private BatchPluginResource createBatchResource(final String moduleCompleteKey,
                                                    final ResourceDescriptor resourceDescriptor,
                                                    final String type,
                                                    final UrlParameters urlParams,
                                                    final BatchedWebResourceDescriptor batchedWebResourceDescriptor)
    {
        final Map<String, String> params = new TreeMap<String, String>();
        for (final String param : BATCH_PARAMS)
        {
            final String value = resourceDescriptor.getParameter(param);
            if (StringUtils.isNotEmpty(value))
            {
                params.put(param, value);
            }
        }

        return new BatchPluginResource(ResourceKey.Builder.lazy(moduleCompleteKey, Suppliers.ofInstance(type)), urlParams, params, batchedWebResourceDescriptor);
    }

    public String getResourceUrl(final String moduleCompleteKey, final String resourceName)
    {
        return webResourceUrlProvider.getResourceUrl(moduleCompleteKey, resourceName);
    }

    public TransformerCache getTransformerCache() {
        return transformerCache;
    }
}
