package com.atlassian.plugin.webresource;

import static com.atlassian.plugin.servlet.AbstractFileServerServlet.PATH_SEPARATOR;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.RESOURCE_URL_PREFIX;
import static com.atlassian.plugin.servlet.AbstractFileServerServlet.SERVLET_PATH;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.webresource.url.CDNSupport;
import com.atlassian.plugin.webresource.url.NamedHash;
import com.atlassian.plugin.webresource.url.UrlParameters;
import com.atlassian.util.concurrent.LazyReference;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Represents a single plugin resource.
 *
 * It provides methods to parse and generate urls to locate a single plugin resource.
 *
 * Note: This PluginResource does not use it's parameters in generating the url. 
 *
 * @since 2.2
 */
public class SinglePluginResource extends AbstractPluginResource
{
    /**
     * The url prefix to a single plugin resource: "/download/resources/"
     */
    public static final String URL_PREFIX = PATH_SEPARATOR + SERVLET_PATH + PATH_SEPARATOR + RESOURCE_URL_PREFIX;

    private final String resourceName;
    private final String moduleCompleteKey;
    private final boolean cached;
    private final LazyReference<String> type;
    
    public SinglePluginResource(final String resourceName, final String moduleCompleteKey, final boolean cached, final CDNSupport cdn)
    {
        this(resourceName, moduleCompleteKey, cached, UrlParameters.of(cdn), Collections.<String, String>emptyMap());
    }

    public SinglePluginResource(final String resourceName, final String moduleCompleteKey, final boolean cached, final UrlParameters urlParams, final Map<String, String> params)
    {
        super(urlParams, params, Collections.<BatchedWebResourceDescriptor>emptySet());
        this.resourceName = resourceName;
        this.moduleCompleteKey = moduleCompleteKey;
        this.cached = cached;
        this.type = new LazyReference<String>() {
            @Override
            protected String create() throws Exception
            {
                return ResourceUtils.getType(resourceName);
            }
        };
    }

    @Override
    public String getResourceName()
    {
        return resourceName;
    }

    @Override
    public String getModuleCompleteKey()
    {
        return moduleCompleteKey;
    }

    @Override
    public String getVersion(WebResourceIntegration integration)
    {
        final Plugin plugin = integration.getPluginAccessor().getEnabledPluginModule(getModuleCompleteKey()).getPlugin();
        return plugin.getPluginInformation().getVersion();
    }

    @Override
    public boolean isCacheSupported()
    {
        return cached;
    }
    
    /**
     * Returns a url string in the format: /download/resources/MODULE_COMPLETE_KEY/RESOURCE_NAME
     *
     * e.g. /download/resources/example.plugin:webresources/foo.css
     */
    @Override
    public String getUrl()
    {
        final StringBuilder sb = new StringBuilder();
        sb.append(URL_PREFIX).append(PATH_SEPARATOR).append(moduleCompleteKey).append(PATH_SEPARATOR).append(
                resourceName);
        ResourceUtils.addParamsToUrl(sb, getBatchableParams());
        return sb.toString();
    }

    @Override
    public String getType()
    {
        return type.get();
    }
}
