package com.atlassian.plugin.webresource;

import com.atlassian.fugue.Pair;
import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.servlet.DownloadableResource;
import com.atlassian.plugin.webresource.transformer.TransformableResource;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.atlassian.plugin.webresource.transformer.TransformerParameters;
import com.atlassian.plugin.webresource.transformer.UrlReadingWebResourceTransformerModuleDescriptor;
import com.atlassian.plugin.webresource.transformer.WebResourceTransformerModuleDescriptor;
import com.atlassian.plugin.webresource.url.UrlBuilder;
import org.apache.commons.lang.Validate;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;


/**
 * Represents a set of transformer invocations for a specific web resource set and extension.  Transformers are retrieved
 * from the plugin system on request, not plugin initialisation, since plugin start order is indeterminate.
 *
 * @since 2.5.0
 */
public class WebResourceTransformation
{
    private final String extension;
    private final String type;
    private final Map<String, Element> transformerElements;
    private final Iterable<String> transformerKeys;
    private Logger log = LoggerFactory.getLogger(WebResourceTransformation.class);

    public WebResourceTransformation(Element element)
    {
        Validate.notNull(element.attribute("extension"));

        this.type = element.attributeValue("extension");
        this.extension = "." + type;

        LinkedHashMap<String, Element> transformers = new LinkedHashMap<String, Element>();
        for (Element transformElement : (List<Element>)element.elements("transformer"))
        {
            transformers.put(transformElement.attributeValue("key"), transformElement);
        }
        transformerElements = Collections.unmodifiableMap(transformers);
        transformerKeys = transformerElements.keySet();
    }

    public boolean matches(ResourceLocation location)
    {
        String loc = location.getLocation();
        if (loc == null || "".equals(loc.trim()))
        {
            loc = location.getName();
        }
        return loc.endsWith(extension);
    }

    public boolean matches(String location)
    {
        return location.equals(type);
    }

    public void addTransformParameters(TransformerCache transformerCache,
                                       WebResourceModuleDescriptor webResourceModuleDescriptor, UrlBuilder urlBuilder)
    {
        for (String key : transformerKeys)
        {
            UrlReadingWebResourceTransformerModuleDescriptor descriptor = transformerCache.getDescriptor(key);
            if (descriptor != null)
            {
                descriptor.getModule().makeUrlBuilder(TransformerParameters.of(webResourceModuleDescriptor)).addToUrl(urlBuilder);
            }
        }
    }

    public boolean containsOnlyPureUrlReadingTransformers(TransformerCache transformerCache)
    {
        for (String key : transformerKeys)
        {
            WebResourceTransformerModuleDescriptor deprecatedTransformer = transformerCache.getDeprecatedDescriptor(key);
            if (deprecatedTransformer != null)
            {
                return false;
            }
        }
        return true;
    }

    public Iterable<WebResourceTransformerModuleDescriptor> getDeprecatedTransformers(TransformerCache transformerCache)
    {
        List<WebResourceTransformerModuleDescriptor> deprecatedTransformers = new LinkedList<WebResourceTransformerModuleDescriptor>();
        for (String key : transformerKeys)
        {
            WebResourceTransformerModuleDescriptor deprecatedTransformer = transformerCache.getDeprecatedDescriptor(key);
            if (deprecatedTransformer != null)
            {
                deprecatedTransformers.add(deprecatedTransformer);
            }
        }
        return deprecatedTransformers;
    }

    public DownloadableResource transformDownloadableResource(TransformerCache transformerCache, WebResourceModuleDescriptor webResourceModuleDescriptor, DownloadableResource resource, ResourceLocation resourceLocation, String filePath, QueryParams params)
    {
        DownloadableResource lastResource = resource;
        for (String transformerKey : transformerKeys)
        {
            UrlReadingWebResourceTransformerModuleDescriptor descriptor = transformerCache.getDescriptor(transformerKey);
            if (descriptor != null)
            {
                TransformableResource transformableResource = new TransformableResource(resourceLocation, filePath, lastResource);
                lastResource = descriptor.getModule().makeResourceTransformer(TransformerParameters.of(webResourceModuleDescriptor)).transform(transformableResource, params);
            }
            else
            {
                WebResourceTransformerModuleDescriptor deprecatedDescriptor = transformerCache.getDeprecatedDescriptor(transformerKey);
                if (deprecatedDescriptor != null)
                {
                    lastResource = deprecatedDescriptor.getModule().transform(transformerElements.get(transformerKey), resourceLocation, filePath, lastResource);
                }
                else
                {
                    log.warn("Web resource transformer {} not found for resource {}, skipping", transformerKey, resourceLocation.getName());
                }
            }
        }

        return lastResource;
    }
}