package com.atlassian.plugin.webresource.assembler;

import com.atlassian.plugin.webresource.BatchResourceContentsWebFormatter;
import com.atlassian.plugin.webresource.JavascriptWebResource;
import com.atlassian.plugin.webresource.PluginResource;
import com.atlassian.plugin.webresource.ResourceBatchingConfiguration;
import com.atlassian.plugin.webresource.WebResourceFormatter;
import com.atlassian.plugin.webresource.WebResourceIntegration;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.atlassian.plugin.webresource.data.DataTagWriter;
import com.atlassian.webresource.api.UrlMode;
import com.atlassian.webresource.api.assembler.WebResource;
import com.atlassian.webresource.api.assembler.WebResourceSet;
import com.atlassian.webresource.api.assembler.resource.PluginUrlResource;
import com.atlassian.webresource.api.data.PluginDataResource;
import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Implementation of WebResourceSet
 * @since v3.0
 */
public class DefaultWebResourceSet implements WebResourceSet
{
    private static final Logger log = LoggerFactory.getLogger(DefaultWebResourceSet.class);

    private final ResourceBatchingConfiguration batchingConfiguration;
    private final WebResourceUrlProvider webResourceUrlProvider;
    private final WebResourceIntegration webResourceIntegration;
    private final Iterable<PluginResourceContainer> resources;
    private final Iterable<PluginDataResource> data;

    public DefaultWebResourceSet(ResourceBatchingConfiguration batchingConfiguration,
         WebResourceUrlProvider webResourceUrlProvider, WebResourceIntegration webResourceIntegration,
         Iterable<PluginResourceContainer> resources, Iterable<PluginDataResource> data)
    {
        this.batchingConfiguration = batchingConfiguration;
        this.webResourceIntegration = webResourceIntegration;
        this.webResourceUrlProvider = webResourceUrlProvider;
        this.resources = resources;
        this.data = data;
    }

    @Override
    public Iterable<WebResource> getResources()
    {
        Collection<WebResource> webResources = Lists.<WebResource>newLinkedList(data);
        for (PluginResourceContainer container : resources)
        {
            webResources.add(container.pluginUrlResource);
        }
        return webResources;
    }

    @Override
    public <T extends WebResource> Iterable<T> getResources(Class<T> clazz)
    {
        return (Iterable<T>) Iterables.filter(getResources(), Predicates.instanceOf(clazz));
    }

    @Override
    public void writeHtmlTags(Writer writer, UrlMode urlMode)
    {
        writeHtmlTags(writer, urlMode, Predicates.<WebResource>alwaysTrue(), Predicates.<PluginResource>alwaysTrue());
    }

    @Override
    public void writeHtmlTags(Writer writer, UrlMode urlMode, final Predicate<WebResource> predicate)
    {
        writeHtmlTags(writer, urlMode, predicate, Predicates.<PluginResource>alwaysTrue());
    }

    public void writeHtmlTags(Writer writer, UrlMode urlMode, final Predicate<WebResource> predicate, final Predicate<PluginResource> legacyPredicate)
    {
        try
        {
            new DataTagWriter().write(writer, Iterables.filter(data, predicate));
        }
        catch (IOException ex)
        {
            log.error("IOException encountered rendering data tags", ex);
        }

        List<PluginResourceContainer> localCopyOfResources = Lists.newArrayList(Iterables.filter(resources, new Predicate<PluginResourceContainer>()
        {
            @Override
            public boolean apply(@Nullable PluginResourceContainer container)
            {
                return predicate.apply(container.pluginUrlResource) && legacyPredicate.apply(container.pluginResource);
            }
        }));

        boolean enableDeferJsAttribute = webResourceIntegration.isDeferJsAttributeEnabled();

        for (final WebResourceFormatter formatter : WebResourceFormatter.webResourceFormatters)
        {
            for (final Iterator<PluginResourceContainer> iter = localCopyOfResources.iterator(); iter.hasNext();)
            {
                final PluginResourceContainer container = iter.next();
                if (formatter.matches(container.pluginResource.getResourceName()))
                {
                    writeResourceTag(urlMode, container, formatter, writer, enableDeferJsAttribute);
                    iter.remove();
                }
            }
        }

        for (final PluginResourceContainer container : localCopyOfResources)
        {
            writeContentAndSwallowErrors(writer, "<!-- Error loading resource \"", container.pluginResource.getModuleCompleteKey(),
                    "\".  No resource formatter matches \"", container.pluginResource.getResourceName(), "\" -->\n");
        }
    }

    private void writeResourceTag(final UrlMode urlMode, final PluginResourceContainer container, final WebResourceFormatter formatter, final Writer writer, boolean enableDeferJsAttribute)
    {
        String formattedResource;
        if (formatter instanceof JavascriptWebResource)
        {
            formattedResource = ((JavascriptWebResource) formatter).formatResource(container.pluginUrlResource.getStaticUrl(urlMode), container.pluginUrlResource.getParams().all(), enableDeferJsAttribute);
        }
        else
        {
            formattedResource = formatter.formatResource(container.pluginUrlResource.getStaticUrl(urlMode), container.pluginUrlResource.getParams().all());
        }

        // insert the dependency information into the formatted resource
        if (batchingConfiguration.isBatchContentTrackingEnabled())
        {
            formattedResource = BatchResourceContentsWebFormatter.insertBatchResourceContents(container.pluginResource, formattedResource);
        }

        writeContentAndSwallowErrors(writer, formattedResource);
    }

    private void writeContentAndSwallowErrors(final Writer writer, final String... contents)
    {
        try
        {
            for (final String content : contents)
            {
                writer.write(content);
            }
        }
        catch (final IOException ex)
        {
            log.error("IOException encountered rendering resource", ex);
        }
    }

    /**
     * Private, local {@link PluginUrlResource} that exposes the underlying {@link PluginResource}
     */
    private static class PluginResourceContainer
    {
        private final PluginResource pluginResource;
        private final PluginUrlResource pluginUrlResource;

        private PluginResourceContainer(WebResourceIntegration webResourceIntegration, WebResourceUrlProvider webResourceUrlProvider, PluginResource pluginResource)
        {
            this.pluginResource = pluginResource;
            this.pluginUrlResource = PluginUrlResourceFactory.create(webResourceIntegration, webResourceUrlProvider, pluginResource);
        }
    }

    public static Builder builder(final WebResourceUrlProvider webResourceUrlProvider,
                                  final WebResourceIntegration webResourceIntegration)
    {
        return new Builder(webResourceUrlProvider, webResourceIntegration);
    }

    public static class Builder
    {
        private final WebResourceUrlProvider webResourceUrlProvider;
        private final WebResourceIntegration webResourceIntegration;
        private Iterable<PluginResourceContainer> resources;
        private Iterable<PluginDataResource> data;

        public Builder(final WebResourceUrlProvider webResourceUrlProvider,
                       final WebResourceIntegration webResourceIntegration)
        {
            this.webResourceUrlProvider = webResourceUrlProvider;
            this.webResourceIntegration = webResourceIntegration;
        }

        public Builder(final Builder that)
        {
            this.webResourceUrlProvider = that.webResourceUrlProvider;
            this.webResourceIntegration = that.webResourceIntegration;
            this.resources = that.resources;
            this.data = that.data;
        }

        public Builder copy()
        {
            return new Builder(this);
        }

        public Builder resources(Iterable<PluginResource> resources)
        {
            this.resources = createPluginResourceContainers(resources);
            return this;
        }

        public Builder data(Iterable<PluginDataResource> data)
        {
            this.data = data;
            return this;
        }

        public WebResourceSet build(final ResourceBatchingConfiguration batchingConfiguration)
        {
            return new DefaultWebResourceSet(batchingConfiguration, webResourceUrlProvider, webResourceIntegration,
                resources, data);
        }

        /**
         * Maps a list of {@link PluginResource} to {@link PluginResourceContainer}
         * @param resources resources
         * @return list of LocalPluginUrlResource
         */
        private Iterable<PluginResourceContainer> createPluginResourceContainers(Iterable<PluginResource> resources)
        {
            return Iterables.transform(resources, new Function<PluginResource, PluginResourceContainer>()
            {
                @Override
                public PluginResourceContainer apply(@Nullable PluginResource pluginResource)
                {
                    return new PluginResourceContainer(webResourceIntegration, webResourceUrlProvider, pluginResource);
                }
            });
        }
    }
}
