package com.atlassian.plugin.webresource.assembler;

import com.google.common.collect.Sets;

import java.util.Set;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Mutable representation of the current state of resources included.
 * @since 3.2
 */
class InclusionState
{
    private boolean superbatch;
    private Set<String> webresources;
    private Set<String> contexts;
    private Set<String> dataKeys;

    public InclusionState(boolean superbatch, Set<String> webresources, Set<String> contexts, Set<String> dataKeys)
    {
        checkNotNull(webresources);
        checkNotNull(contexts);
        checkNotNull(dataKeys);

        this.superbatch = superbatch;
        this.webresources = webresources;
        this.contexts = contexts;
        this.dataKeys = dataKeys;
    }

    public InclusionState copy()
    {
        return new InclusionState(superbatch, Sets.newHashSet(webresources), Sets.newHashSet(contexts),
                Sets.newHashSet(dataKeys));
    }

    public void reset(InclusionState that)
    {
        this.superbatch = that.superbatch;
        this.webresources = Sets.newHashSet(that.webresources);
        this.contexts = Sets.newHashSet(that.contexts);
        this.dataKeys = Sets.newHashSet(that.dataKeys);
    }

    /**
     * Webresources that have been included in previous calls to includeResources, and all the individual resources
     * in included contexts. Mutable!
     */
    public Set<String> getWebresources()
    {
        return webresources;
    }

    /**
     * Webresource contexts that have been included in previous calls to includeResources. Mutable!
     */
    public Set<String> getContexts()
    {
        return contexts;
    }

    /**
     * Data keys that have been included in previous calls to includeData. Mutable!
     */
    public Set<String> getDataKeys()
    {
        return dataKeys;
    }

    /**
     * Has the superbatch been included.
     */
    public boolean isSuperbatch()
    {
        return superbatch;
    }

    public void setSuperbatch(boolean superbatch)
    {
        this.superbatch = superbatch;
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        InclusionState that = (InclusionState) o;

        if (superbatch != that.superbatch) return false;
        if (!contexts.equals(that.contexts)) return false;
        if (!dataKeys.equals(that.dataKeys)) return false;
        if (!webresources.equals(that.webresources)) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        int result = (superbatch ? 1 : 0);
        result = 31 * result + webresources.hashCode();
        result = 31 * result + contexts.hashCode();
        result = 31 * result + dataKeys.hashCode();
        return result;
    }
}
