package com.atlassian.plugin.webresource.condition;

import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * Tracks the state of any conditions run, the set of transformations performed, and the set of descriptors included.
* @since 3.2
*/
public class ConditionState
{
    private final Map<DecoratingCondition, Boolean> conditionValues;
    private final Set<WebResourceModuleDescriptor> webResourceModuleDescriptors;
    private final Set<String> resourceTypes;

    public ConditionState()
    {
        this.conditionValues = new HashMap<DecoratingCondition, Boolean>();
        this.webResourceModuleDescriptors = new HashSet<WebResourceModuleDescriptor>();
        this.resourceTypes = new HashSet<String>();
    }

    private ConditionState(ConditionState that)
    {
        this.conditionValues = new HashMap<DecoratingCondition, Boolean>(that.conditionValues);
        this.webResourceModuleDescriptors = new HashSet<WebResourceModuleDescriptor>(that.webResourceModuleDescriptors);
        this.resourceTypes = new HashSet<String>(that.resourceTypes);
    }

    public ConditionState copy()
    {
        return new ConditionState(this);
    }

    public Boolean getConditionResult(DecoratingCondition condition)
    {
        return conditionValues.get(condition);
    }

    public Set<DecoratingCondition> getConditions()
    {
        return conditionValues.keySet();
    }

    public Set<WebResourceModuleDescriptor> getWebResourceModuleDescriptors()
    {
        return webResourceModuleDescriptors;
    }

    public Set<String> getResourceTypes()
    {
        return resourceTypes;
    }

    public void addCondition(DecoratingCondition condition, boolean value)
    {
        conditionValues.put(condition, value);
    }

    public void addAll(ConditionState that)
    {
        this.conditionValues.putAll(that.conditionValues);
    }

    public void addWebResourceModuleDescriptor(WebResourceModuleDescriptor webResourceModuleDescriptor)
    {
        webResourceModuleDescriptors.add(webResourceModuleDescriptor);
    }

    public void addResourceType(String resourceType)
    {
        this.resourceTypes.add(resourceType);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ConditionState that = (ConditionState) o;

        if (!conditionValues.equals(that.conditionValues)) return false;
        if (!webResourceModuleDescriptors.equals(that.webResourceModuleDescriptors)) return false;
        if (!resourceTypes.equals(that.resourceTypes)) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        int result = conditionValues.hashCode();
        result = 31 * result + webResourceModuleDescriptors.hashCode();
        result = 31 * result + resourceTypes.hashCode();
        return result;
    }
}
