package com.atlassian.plugin.webresource.url;

import com.atlassian.plugin.webresource.WebResourceModuleDescriptor;
import com.atlassian.plugin.webresource.condition.ConditionState;
import com.atlassian.plugin.webresource.transformer.StaticTransformers;
import com.atlassian.plugin.webresource.transformer.TransformerCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.Map;

/**
 * Lazily constructs {@link com.atlassian.plugin.webresource.url.DefaultUrlBuilder} instances for a given type.
 *
 * @since v3.0.5
 */
public class DefaultUrlBuilderMap
{
    private final Iterable<WebResourceModuleDescriptor> wrmds;
    private final StaticTransformers staticTransformers;
    private final Map<String, UrlParameters> urlBuildersByType;
    private final TransformerCache transformerCache;

    public DefaultUrlBuilderMap(WebResourceModuleDescriptor wrmd, TransformerCache transformerCache, StaticTransformers staticTransformers)
    {
        this.staticTransformers = staticTransformers;
        this.wrmds = ImmutableList.of(wrmd);
        this.transformerCache = transformerCache;
        this.urlBuildersByType = Maps.newHashMap();
    }

    public DefaultUrlBuilderMap(Iterable<WebResourceModuleDescriptor> wrmds, TransformerCache transformerCache, StaticTransformers staticTransformers)
    {
        this.staticTransformers = staticTransformers;
        this.wrmds = Lists.newArrayList(wrmds);
        this.transformerCache = transformerCache;
        this.urlBuildersByType = Maps.newHashMap();
    }

    /**
     * Returns a UrlParameters for the given type.
     * If none has been created for the given type, one is initialised by calling
     * {@link com.atlassian.plugin.webresource.WebResourceModuleDescriptor#addToUrl(String, com.atlassian.plugin.webresource.transformer.StaticTransformers, com.atlassian.plugin.webresource.transformer.TransformerCache, DefaultUrlBuilder, com.atlassian.plugin.webresource.condition.ConditionState)}
     * for each WebResourceModuleDescriptor that this url builder is aware of.
     * @param type resource type (eg js / css)
     * @param conditionsRun
     * @return UrlParameters for the given type
     */
    public UrlParameters getOrCreateForType(String type, ConditionState conditionsRun)
    {
        UrlParameters urlParams = urlBuildersByType.get(type);
        if (null == urlParams)
        {
            DefaultUrlBuilder urlBuilder = new DefaultUrlBuilder();
            for (WebResourceModuleDescriptor wrmd : wrmds)
            {
                wrmd.addToUrl(type, staticTransformers, transformerCache, urlBuilder, conditionsRun);
            }
            urlParams = urlBuilder.build();
            urlBuildersByType.put(type, urlParams);
        }
        return urlParams;
    }
}
