package com.atlassian.plugin.webresource.url;

import com.atlassian.plugin.webresource.SuperBatchPluginResource;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;

import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * Immutable container for url parameters
 *
 * @since v3.0.5
 */
public class UrlParameters
{
    private static final UrlParameters EMPTY_CDN_OK = new UrlParameters(CDNSupport.OK);
    private static final UrlParameters EMPTY_CDN_TAINTED = new UrlParameters(CDNSupport.Tainted);

    private final List<NamedHash> hashes;
    private final Map<String, String> queryString;
    private final CDNSupport cdn;

    public static UrlParameters of(CDNSupport cdn)
    {
        switch (cdn)
        {
            case OK: return EMPTY_CDN_OK;
            case Tainted: return EMPTY_CDN_TAINTED;
            default: throw new AssertionError("Unhandled CDNSupport parameter: " + cdn);
        }
    }

    public static UrlParameters of(CDNSupport cdn, List<NamedHash> hashes, Map<String, String> queryString)
    {
        return new UrlParameters(cdn, hashes, queryString);
    }

    private UrlParameters(CDNSupport cdn)
    {
        this(cdn, Collections.<NamedHash>emptyList(), Collections.<String, String>emptyMap());
    }

    private UrlParameters(CDNSupport cdn, List<NamedHash> hashes, Map<String, String> queryString)
    {
        this.cdn = cdn;
        this.hashes = ImmutableList.copyOf(hashes);
        this.queryString = ImmutableMap.copyOf(queryString);
    }

    public List<NamedHash> allHashes()
    {
        return hashes;
    }

    public Map<String, String> toQueryString()
    {
        return queryString;
    }

    public CDNSupport supportsCdn()
    {
        return cdn;
    }

    public UrlParameters merge(UrlParameters other)
    {
        Map<String, String> mergedQueryString = Maps.newHashMap(queryString);
        mergedQueryString.putAll(other.queryString);
        List<NamedHash> mergedHashes = ImmutableList.copyOf(Iterables.concat(hashes, other.hashes));
        CDNSupport mergedCdn = cdn.combine(other.cdn);
        return new UrlParameters(mergedCdn, mergedHashes, mergedQueryString);
    }

    @Override
    public boolean equals(Object o)
    {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        UrlParameters that = (UrlParameters) o;

        if (cdn != that.cdn) return false;
        if (!hashes.equals(that.hashes)) return false;
        if (!queryString.equals(that.queryString)) return false;

        return true;
    }

    @Override
    public int hashCode()
    {
        int result = hashes.hashCode();
        result = 31 * result + queryString.hashCode();
        result = 31 * result + cdn.hashCode();
        return result;
    }
}
